# 🚀 Deployment Guide - Sistem Manajemen Anggota PKS

## 📋 Overview

Panduan lengkap untuk deployment **Sistem Manajemen Anggota PKS** ke lingkungan production. Panduan ini mencakup persiapan server, konfigurasi, dan optimasi untuk performa maksimal.

## 🎯 Prerequisites

### Server Requirements
- **OS**: Ubuntu 20.04+ / CentOS 8+ / Debian 11+
- **RAM**: Minimum 2GB, Recommended 4GB+
- **Storage**: Minimum 20GB SSD
- **CPU**: 2+ cores
- **Network**: Stable internet connection

### Software Requirements
- **PHP**: 8.2 atau lebih tinggi
- **MySQL**: 8.0+ atau MariaDB 10.6+
- **Nginx**: 1.18+ atau Apache 2.4+
- **Composer**: Latest version
- **Node.js**: 16.x+ dan NPM 8.x+
- **SSL Certificate**: Let's Encrypt atau commercial

## 🏗️ Server Setup

### 1. Update System
```bash
# Ubuntu/Debian
sudo apt update && sudo apt upgrade -y

# CentOS/RHEL
sudo yum update -y
```

### 2. Install PHP 8.2
```bash
# Ubuntu/Debian
sudo apt install software-properties-common
sudo add-apt-repository ppa:ondrej/php
sudo apt update
sudo apt install php8.2 php8.2-cli php8.2-fpm php8.2-mysql php8.2-xml php8.2-mbstring php8.2-curl php8.2-zip php8.2-gd php8.2-bcmath php8.2-intl php8.2-xmlrpc php8.2-soap

# CentOS/RHEL
sudo dnf install epel-release
sudo dnf install https://rpms.remirepo.net/enterprise/remi-release-8.rpm
sudo dnf module enable php:remi-8.2
sudo dnf install php php-cli php-fpm php-mysqlnd php-xml php-mbstring php-curl php-zip php-gd php-bcmath php-intl
```

### 3. Install MySQL
```bash
# Ubuntu/Debian
sudo apt install mysql-server mysql-client

# CentOS/RHEL
sudo dnf install mysql-server mysql

# Start and enable MySQL
sudo systemctl start mysql
sudo systemctl enable mysql

# Secure MySQL installation
sudo mysql_secure_installation
```

### 4. Install Nginx
```bash
# Ubuntu/Debian
sudo apt install nginx

# CentOS/RHEL
sudo dnf install nginx

# Start and enable Nginx
sudo systemctl start nginx
sudo systemctl enable nginx
```

### 5. Install Composer
```bash
# Download and install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer
sudo chmod +x /usr/local/bin/composer

# Verify installation
composer --version
```

### 6. Install Node.js
```bash
# Using NodeSource repository
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs

# Verify installation
node --version
npm --version
```

## 🗄️ Database Setup

### 1. Create Database
```sql
-- Login to MySQL
mysql -u root -p

-- Create database
CREATE DATABASE bkdpd CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Create user
CREATE USER 'bkdpd_user'@'localhost' IDENTIFIED BY 'strong_password_here';

-- Grant privileges
GRANT ALL PRIVILEGES ON bkdpd.* TO 'bkdpd_user'@'localhost';
FLUSH PRIVILEGES;

-- Exit MySQL
EXIT;
```

### 2. Optimize MySQL Configuration
```bash
# Edit MySQL configuration
sudo nano /etc/mysql/mysql.conf.d/mysqld.cnf

# Add these optimizations
[mysqld]
innodb_buffer_pool_size = 1G
max_connections = 200
query_cache_size = 64M
query_cache_type = 1
innodb_log_file_size = 256M
innodb_flush_log_at_trx_commit = 2
innodb_flush_method = O_DIRECT
```

### 3. Restart MySQL
```bash
sudo systemctl restart mysql
```

## 📁 Application Deployment

### 1. Clone Repository
```bash
# Create application directory
sudo mkdir -p /var/www/bkdpd
sudo chown -R $USER:$USER /var/www/bkdpd

# Clone repository
cd /var/www/bkdpd
git clone <repository-url> .

# Or upload files via SCP/SFTP
```

### 2. Install Dependencies
```bash
# Install PHP dependencies
composer install --no-dev --optimize-autoloader

# Install Node.js dependencies
npm install

# Build assets
npm run build
```

### 3. Environment Configuration
```bash
# Copy environment file
cp .env.example .env

# Generate application key
php artisan key:generate

# Edit environment file
nano .env
```

### 4. Environment Variables
```env
# Application
APP_NAME="Sistem Manajemen Anggota PKS"
APP_ENV=production
APP_KEY=base64:your-generated-key
APP_DEBUG=false
APP_URL=https://your-domain.com

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=bkdpd
DB_USERNAME=bkdpd_user
DB_PASSWORD=strong_password_here

# Cache
CACHE_DRIVER=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis

# Redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

# Mail
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@your-domain.com
MAIL_FROM_NAME="Sistem PKS"

# File Storage
FILESYSTEM_DISK=local
```

### 5. Database Migration
```bash
# Run migrations
php artisan migrate --force

# Seed database
php artisan db:seed --force
```

### 6. Storage Setup
```bash
# Create storage link
php artisan storage:link

# Set permissions
sudo chown -R www-data:www-data /var/www/bkdpd
sudo chmod -R 775 /var/www/bkdpd/storage
sudo chmod -R 775 /var/www/bkdpd/bootstrap/cache
```

### 7. Cache Optimization
```bash
# Clear caches
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

# Optimize for production
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

## 🌐 Web Server Configuration

### Nginx Configuration

#### 1. Create Site Configuration
```bash
sudo nano /etc/nginx/sites-available/bkdpd
```

#### 2. Nginx Configuration
```nginx
server {
    listen 80;
    server_name your-domain.com www.your-domain.com;
    root /var/www/bkdpd/public;
    index index.php index.html;

    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header Referrer-Policy "no-referrer-when-downgrade" always;
    add_header Content-Security-Policy "default-src 'self' http: https: data: blob: 'unsafe-inline'" always;

    # Gzip compression
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_proxied expired no-cache no-store private must-revalidate auth;
    gzip_types text/plain text/css text/xml text/javascript application/x-javascript application/xml+rss application/javascript;

    # Handle Laravel routes
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    # PHP handling
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
        fastcgi_hide_header X-Powered-By;
    }

    # Static files caching
    location ~* \.(jpg|jpeg|png|gif|ico|css|js|woff|woff2|ttf|svg)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
        access_log off;
    }

    # Security - deny access to sensitive files
    location ~ /\. {
        deny all;
    }

    location ~ /(storage|bootstrap/cache) {
        deny all;
    }

    # Rate limiting
    limit_req_zone $binary_remote_addr zone=login:10m rate=5r/m;
    limit_req_zone $binary_remote_addr zone=api:10m rate=30r/m;

    location /login {
        limit_req zone=login burst=3 nodelay;
        try_files $uri $uri/ /index.php?$query_string;
    }

    location /api/ {
        limit_req zone=api burst=10 nodelay;
        try_files $uri $uri/ /index.php?$query_string;
    }
}
```

#### 3. Enable Site
```bash
# Enable site
sudo ln -s /etc/nginx/sites-available/bkdpd /etc/nginx/sites-enabled/

# Test configuration
sudo nginx -t

# Reload Nginx
sudo systemctl reload nginx
```

### Apache Configuration (Alternative)

#### 1. Enable Required Modules
```bash
sudo a2enmod rewrite
sudo a2enmod ssl
sudo a2enmod headers
```

#### 2. Create Virtual Host
```bash
sudo nano /etc/apache2/sites-available/bkdpd.conf
```

#### 3. Apache Configuration
```apache
<VirtualHost *:80>
    ServerName your-domain.com
    ServerAlias www.your-domain.com
    DocumentRoot /var/www/bkdpd/public
    
    <Directory /var/www/bkdpd/public>
        AllowOverride All
        Require all granted
    </Directory>
    
    # Security headers
    Header always set X-Frame-Options "SAMEORIGIN"
    Header always set X-XSS-Protection "1; mode=block"
    Header always set X-Content-Type-Options "nosniff"
    Header always set Referrer-Policy "no-referrer-when-downgrade"
    
    # Gzip compression
    LoadModule deflate_module modules/mod_deflate.so
    <Location />
        SetOutputFilter DEFLATE
        SetEnvIfNoCase Request_URI \
            \.(?:gif|jpe?g|png)$ no-gzip dont-vary
        SetEnvIfNoCase Request_URI \
            \.(?:exe|t?gz|zip|bz2|sit|rar)$ no-gzip dont-vary
    </Location>
    
    ErrorLog ${APACHE_LOG_DIR}/bkdpd_error.log
    CustomLog ${APACHE_LOG_DIR}/bkdpd_access.log combined
</VirtualHost>
```

## 🔒 SSL Certificate

### Using Let's Encrypt
```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx

# Get SSL certificate
sudo certbot --nginx -d your-domain.com -d www.your-domain.com

# Auto-renewal
sudo crontab -e
# Add this line:
0 12 * * * /usr/bin/certbot renew --quiet
```

### Manual SSL Certificate
```bash
# Upload certificate files
sudo mkdir -p /etc/ssl/certs/bkdpd
sudo nano /etc/ssl/certs/bkdpd/your-domain.crt
sudo nano /etc/ssl/private/bkdpd/your-domain.key

# Update Nginx configuration for SSL
sudo nano /etc/nginx/sites-available/bkdpd
```

## ⚡ Performance Optimization

### 1. Install Redis
```bash
# Ubuntu/Debian
sudo apt install redis-server

# CentOS/RHEL
sudo dnf install redis

# Start and enable Redis
sudo systemctl start redis
sudo systemctl enable redis
```

### 2. Install Supervisor
```bash
# Ubuntu/Debian
sudo apt install supervisor

# CentOS/RHEL
sudo dnf install supervisor

# Start and enable Supervisor
sudo systemctl start supervisord
sudo systemctl enable supervisord
```

### 3. Configure Queue Worker
```bash
# Create supervisor configuration
sudo nano /etc/supervisor/conf.d/bkdpd-worker.conf
```

```ini
[program:bkdpd-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/bkdpd/artisan queue:work redis --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/bkdpd/storage/logs/worker.log
stopwaitsecs=3600
```

### 4. Start Queue Worker
```bash
# Reload supervisor
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start bkdpd-worker:*
```

### 5. Configure Cron Jobs
```bash
# Edit crontab
sudo crontab -e

# Add Laravel scheduler
* * * * * cd /var/www/bkdpd && php artisan schedule:run >> /dev/null 2>&1

# Add queue restart (optional)
*/5 * * * * cd /var/www/bkdpd && php artisan queue:restart >> /dev/null 2>&1
```

## 🔧 Monitoring & Logging

### 1. Configure Log Rotation
```bash
sudo nano /etc/logrotate.d/bkdpd
```

```
/var/www/bkdpd/storage/logs/*.log {
    daily
    missingok
    rotate 14
    compress
    notifempty
    create 644 www-data www-data
    postrotate
        /bin/kill -USR1 `cat /var/run/nginx.pid 2>/dev/null` 2>/dev/null || true
    endscript
}
```

### 2. Install Monitoring Tools
```bash
# Install htop for system monitoring
sudo apt install htop

# Install iotop for disk monitoring
sudo apt install iotop

# Install nethogs for network monitoring
sudo apt install nethogs
```

### 3. Configure Firewall
```bash
# Install UFW
sudo apt install ufw

# Configure firewall
sudo ufw default deny incoming
sudo ufw default allow outgoing
sudo ufw allow ssh
sudo ufw allow 'Nginx Full'
sudo ufw allow 3306  # MySQL (if needed externally)
sudo ufw enable
```

## 🚀 Deployment Script

### 1. Create Deployment Script
```bash
nano deploy.sh
```

```bash
#!/bin/bash

# Deployment script for BKD PD
set -e

echo "🚀 Starting deployment..."

# Pull latest changes
git pull origin main

# Install/update dependencies
composer install --no-dev --optimize-autoloader
npm install
npm run build

# Run migrations
php artisan migrate --force

# Clear and cache
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

php artisan config:cache
php artisan route:cache
php artisan view:cache

# Restart services
sudo systemctl reload nginx
sudo supervisorctl restart bkdpd-worker:*

echo "✅ Deployment completed successfully!"
```

### 2. Make Script Executable
```bash
chmod +x deploy.sh
```

## 🔄 Backup Strategy

### 1. Database Backup
```bash
# Create backup script
nano backup-db.sh
```

```bash
#!/bin/bash
BACKUP_DIR="/var/backups/bkdpd"
DATE=$(date +%Y%m%d_%H%M%S)
DB_NAME="bkdpd"
DB_USER="bkdpd_user"
DB_PASS="your_password"

mkdir -p $BACKUP_DIR

# Create database backup
mysqldump -u $DB_USER -p$DB_PASS $DB_NAME > $BACKUP_DIR/db_backup_$DATE.sql

# Compress backup
gzip $BACKUP_DIR/db_backup_$DATE.sql

# Keep only last 7 days
find $BACKUP_DIR -name "db_backup_*.sql.gz" -mtime +7 -delete

echo "Database backup completed: db_backup_$DATE.sql.gz"
```

### 2. File Backup
```bash
# Create file backup script
nano backup-files.sh
```

```bash
#!/bin/bash
BACKUP_DIR="/var/backups/bkdpd"
DATE=$(date +%Y%m%d_%H%M%S)
APP_DIR="/var/www/bkdpd"

mkdir -p $BACKUP_DIR

# Create file backup
tar -czf $BACKUP_DIR/files_backup_$DATE.tar.gz -C $APP_DIR storage public

# Keep only last 7 days
find $BACKUP_DIR -name "files_backup_*.tar.gz" -mtime +7 -delete

echo "File backup completed: files_backup_$DATE.tar.gz"
```

### 3. Automated Backup
```bash
# Add to crontab
sudo crontab -e

# Add these lines:
0 2 * * * /var/www/bkdpd/backup-db.sh
0 3 * * * /var/www/bkdpd/backup-files.sh
```

## 🧪 Testing Deployment

### 1. Health Check
```bash
# Test application
curl -I https://your-domain.com

# Test database connection
php artisan tinker
>>> DB::connection()->getPdo();

# Test queue
php artisan queue:work --once
```

### 2. Performance Test
```bash
# Install Apache Bench
sudo apt install apache2-utils

# Test performance
ab -n 1000 -c 10 https://your-domain.com/
```

### 3. Security Test
```bash
# Install SSL test tool
sudo apt install sslscan

# Test SSL
sslscan your-domain.com
```

## 🚨 Troubleshooting

### Common Issues

#### 1. Permission Issues
```bash
# Fix permissions
sudo chown -R www-data:www-data /var/www/bkdpd
sudo chmod -R 775 /var/www/bkdpd/storage
sudo chmod -R 775 /var/www/bkdpd/bootstrap/cache
```

#### 2. Database Connection Issues
```bash
# Check MySQL status
sudo systemctl status mysql

# Check database connection
mysql -u bkdpd_user -p -e "SELECT 1"
```

#### 3. Nginx Issues
```bash
# Test Nginx configuration
sudo nginx -t

# Check Nginx logs
sudo tail -f /var/log/nginx/error.log
```

#### 4. PHP Issues
```bash
# Check PHP-FPM status
sudo systemctl status php8.2-fpm

# Check PHP logs
sudo tail -f /var/log/php8.2-fpm.log
```

## 📊 Monitoring Commands

### System Monitoring
```bash
# Check system resources
htop
df -h
free -h

# Check application logs
tail -f /var/www/bkdpd/storage/logs/laravel.log

# Check Nginx logs
tail -f /var/log/nginx/access.log
tail -f /var/log/nginx/error.log

# Check MySQL logs
sudo tail -f /var/log/mysql/error.log
```

### Application Monitoring
```bash
# Check queue status
php artisan queue:monitor

# Check cache status
php artisan cache:table
php artisan cache:clear

# Check scheduled tasks
php artisan schedule:list
```

## 🎯 Production Checklist

- [ ] Server requirements met
- [ ] PHP 8.2+ installed and configured
- [ ] MySQL 8.0+ installed and optimized
- [ ] Nginx/Apache configured with SSL
- [ ] Application deployed and configured
- [ ] Database migrated and seeded
- [ ] Environment variables set
- [ ] Storage linked and permissions set
- [ ] Cache optimized
- [ ] Queue workers running
- [ ] Cron jobs configured
- [ ] Backup strategy implemented
- [ ] Monitoring configured
- [ ] Firewall configured
- [ ] SSL certificate installed
- [ ] Performance tested
- [ ] Security tested

---

**Sistem Manajemen Anggota PKS siap untuk production! 🚀**
