# CI/CD Setup dengan GitHub Actions

## 📋 Overview

Aplikasi ini sudah dilengkapi dengan CI/CD pipeline menggunakan GitHub Actions yang terdiri dari:

1. **CI (Continuous Integration)** - Otomatis testing & code quality check
2. **CD (Continuous Deployment)** - Otomatis deploy ke server

## 🚀 Fitur CI/CD

### CI Pipeline (`.github/workflows/ci.yml`)
Otomatis berjalan ketika:
- Push ke branch: `main`, `develop`, atau `feature/*`
- Pull Request ke branch: `main` atau `develop`

**Yang dilakukan:**
- ✅ Testing PHP (PHPUnit) di PHP 8.1 & 8.2
- ✅ Build assets (NPM) di Node 18 & 20
- ✅ Code style check dengan Laravel Pint
- ✅ Security vulnerability scan
- ✅ Parallel testing untuk kecepatan maksimal

### CD Pipeline (`.github/workflows/deploy.yml`)
Otomatis deploy ketika:
- Push ke branch `main`
- Manual trigger (workflow_dispatch)

**Yang dilakukan:**
- 📦 Build production assets
- 🚀 Deploy ke server (SSH/FTP/cPanel)
- 🔄 Run migrations
- ⚡ Optimize Laravel (cache configs, routes, views)
- ✅ Health check setelah deployment

## 🔧 Setup Awal

### 1. Enable GitHub Actions
1. Buka repository GitHub: https://github.com/mahrizalit/bkdpd
2. Ke tab **Settings** → **Actions** → **General**
3. Pilih "Allow all actions and reusable workflows"
4. Save

### 2. Setup Secrets untuk Deployment

Ke **Settings** → **Secrets and variables** → **Actions** → **New repository secret**

Tambahkan secrets sesuai metode deployment:

#### A. Untuk Deploy via SSH (VPS/Dedicated Server)
```
SSH_HOST          = your-server-ip.com
SSH_USERNAME      = your-username
SSH_PRIVATE_KEY   = -----BEGIN RSA PRIVATE KEY-----... (private key Anda)
SSH_PORT          = 22
DEPLOY_PATH       = /var/www/html/bkdpd
APP_URL           = https://your-domain.com
```

**Cara generate SSH key:**
```bash
ssh-keygen -t rsa -b 4096 -C "github-actions"
# Simpan private key ke GitHub Secrets
# Copy public key ke server: ~/.ssh/authorized_keys
```

#### B. Untuk Deploy via FTP (Shared Hosting)
```
FTP_SERVER        = ftp.your-hosting.com
FTP_USERNAME      = your-ftp-username
FTP_PASSWORD      = your-ftp-password
FTP_SERVER_DIR    = /public_html/
```

#### C. Untuk Deploy via cPanel
```
CPANEL_HOST       = your-domain.com
CPANEL_USERNAME   = cpanel-username
CPANEL_PASSWORD   = cpanel-password
```

#### D. Optional: CloudFlare Cache Clear
```
CLOUDFLARE_ZONE_ID    = your-zone-id
CLOUDFLARE_API_TOKEN  = your-api-token
```

### 3. Aktifkan Deployment

Edit file `.github/workflows/deploy.yml`:

**Untuk SSH deployment:**
```yaml
- name: Deploy to Server via SSH
  if: true  # ⬅️ Ubah dari false ke true
```

**Untuk FTP deployment:**
```yaml
- name: Deploy to Server via FTP
  if: true  # ⬅️ Ubah dari false ke true
```

**Untuk cPanel deployment:**
```yaml
- name: Deploy to cPanel via Git Pull
  if: true  # ⬅️ Ubah dari false ke true
```

**Untuk health check:**
```yaml
post-deploy-checks:
  if: true  # ⬅️ Ubah dari false ke true
```

### 4. Setup Environment di Server

Pastikan server Anda sudah memiliki:

```bash
# PHP 8.1 atau 8.2
php -v

# Composer
composer -V

# Node.js & NPM
node -v
npm -v

# Git (jika pakai SSH deployment)
git --version
```

**Setup project di server:**
```bash
# Clone repository
cd /var/www/html
git clone git@github.com:mahrizalit/bkdpd.git
cd bkdpd

# Copy environment file
cp .env.example .env
nano .env  # Edit sesuai server production

# Install dependencies
composer install --no-dev --optimize-autoloader
npm ci && npm run build

# Setup Laravel
php artisan key:generate
php artisan migrate --force
php artisan storage:link
php artisan optimize

# Set permissions
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

## 📝 Penggunaan

### Testing Otomatis (CI)

Setiap kali push atau create pull request, GitHub Actions akan:

1. Run semua tests
2. Check code style
3. Build assets
4. Scan vulnerabilities

Lihat hasilnya di tab **Actions** di repository GitHub.

### Deployment Otomatis (CD)

**Auto Deploy:**
```bash
# Push ke main branch akan auto deploy
git checkout main
git merge develop
git push origin main
```

**Manual Deploy:**
1. Buka tab **Actions** di GitHub
2. Pilih workflow "CD - Deploy to Production"
3. Click "Run workflow"
4. Pilih branch `main`
5. Click "Run workflow"

## 📊 Monitoring Deployment

### Cek Status di GitHub
1. Buka tab **Actions**
2. Lihat workflow runs (✅ sukses, ❌ gagal)
3. Klik workflow untuk detail logs

### Cek di Server
```bash
# Cek versi deployed
cd /var/www/html/bkdpd
git log -1

# Cek status aplikasi
php artisan about

# Cek logs
tail -f storage/logs/laravel.log
```

## 🎯 Workflow Deployment yang Direkomendasikan

```
Development (local)
    ↓ push
feature/xxx branch → create PR
    ↓ auto test (CI)
develop branch → merge after review
    ↓ auto test (CI)
main branch → push/merge
    ↓ auto deploy (CD)
Production Server ✅
```

## 🔒 Best Practices

### 1. Branch Protection
Aktifkan di **Settings** → **Branches** → **Add rule**:
- Branch name pattern: `main`
- ✅ Require status checks to pass before merging
- ✅ Require branches to be up to date before merging
- ✅ Require pull request reviews before merging

### 2. Environment Protection
Aktifkan di **Settings** → **Environments** → **production**:
- ✅ Required reviewers (pilih 1-2 orang)
- ✅ Wait timer: 5 minutes (opsional)

### 3. Notifications
Setup notifications untuk deployment:

**Slack:** Tambah step di deploy.yml:
```yaml
- name: Notify Slack
  uses: 8398a7/action-slack@v3
  with:
    status: ${{ job.status }}
    webhook_url: ${{ secrets.SLACK_WEBHOOK }}
```

**Discord:** Tambah step di deploy.yml:
```yaml
- name: Notify Discord
  uses: sarisia/actions-status-discord@v1
  with:
    webhook: ${{ secrets.DISCORD_WEBHOOK }}
```

## 🐛 Troubleshooting

### CI Tests Gagal
```bash
# Test di local dulu
php artisan test

# Fix issues, lalu push lagi
```

### Deployment Gagal - SSH Connection
```bash
# Test SSH connection di local
ssh username@your-server.com

# Pastikan public key sudah di server
cat ~/.ssh/id_rsa.pub  # Copy ini
# Paste ke server: ~/.ssh/authorized_keys
```

### Deployment Gagal - Permission Denied
```bash
# Di server, set ownership
cd /var/www/html
chown -R www-data:www-data bkdpd

# Set permissions
cd bkdpd
chmod -R 775 storage bootstrap/cache
```

### Assets Tidak Terupdate
```bash
# Di server, manual build
cd /var/www/html/bkdpd
npm ci
npm run build
php artisan optimize:clear
```

## 📚 Resources

- [GitHub Actions Documentation](https://docs.github.com/en/actions)
- [Laravel Deployment](https://laravel.com/docs/11.x/deployment)
- [SSH Action](https://github.com/appleboy/ssh-action)
- [FTP Deploy Action](https://github.com/SamKirkland/FTP-Deploy-Action)

## 🔄 Customization

### Ubah Branch untuk Auto Deploy
Edit `.github/workflows/deploy.yml`:
```yaml
on:
  push:
    branches: [ main, production ]  # Tambah branch lain
```

### Ubah PHP Version
Edit `.github/workflows/ci.yml`:
```yaml
matrix:
  php-versions: ['8.2', '8.3']  # Ubah versi
```

### Tambah Testing Database Real (MySQL)
Edit `.github/workflows/ci.yml`:
```yaml
services:
  mysql:
    image: mysql:8.0
    env:
      MYSQL_ROOT_PASSWORD: password
      MYSQL_DATABASE: testing
    ports:
      - 3306:3306
```

### Skip Deployment untuk Commit Tertentu
Tambahkan `[skip ci]` di commit message:
```bash
git commit -m "Update docs [skip ci]"
```

## ✅ Checklist Setup

- [ ] Enable GitHub Actions
- [ ] Tambahkan SSH/FTP secrets
- [ ] Test SSH connection ke server
- [ ] Setup project di server
- [ ] Aktifkan deployment workflow (ubah `if: false` → `if: true`)
- [ ] Test manual deployment
- [ ] Setup branch protection rules
- [ ] Setup environment protection
- [ ] Configure notifications (optional)
- [ ] Update documentation dengan URL production

## 📞 Support

Jika ada masalah dengan CI/CD setup, check:
1. GitHub Actions logs
2. Server logs: `/var/www/html/bkdpd/storage/logs/`
3. Web server logs: `/var/log/nginx/error.log` atau `/var/log/apache2/error.log`

---

**Status:** ✅ CI/CD Ready - Tinggal setup secrets dan aktifkan deployment!

